/**
 * \file sdc_ctl_action.h
 *
 * \brief SDC_CTL : Define types, enums, structs for actions (operations)
 *
 * We use action for an individual operation/command while operation refers
 * to the functionality behind.
 * One operation (e.g. install) can be called more than once from multiple
 * actions
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef __SDC_CTL_ACTION_
#define __SDC_CTL_ACTION_

#include <sdc_keystore_keys.h>
#include <sdc_op_conv.h>

typedef enum action_operations {
    ACTION_INSTALL,
    ACTION_REMOVE,
    ACTION_NON_FORMATTED_WRAP,
    ACTION_NON_FORMATTED_UNWRAP,
    ACTION_FORMATTED_WRAP,
    ACTION_FORMATTED_UNWRAP,
    ACTION_NON_FORMATTED_ENCRYPT,
    ACTION_NON_FORMATTED_DECRYPT,
    ACTION_FORMATTED_ENCRYPT,
    ACTION_FORMATTED_DECRYPT,
    ACTION_NON_FORMATTED_SIGN,
    ACTION_NON_FORMATTED_VERIFY,
    ACTION_FORMATTED_SIGN,
    ACTION_FORMATTED_VERIFY,
    ACTION_NON_FORMATTED_DGST,
    ACTION_RNG,
    ACTION_ARCH_INFO
} action_operations_t;

typedef enum action_kid_source {
    ACTION_KEY_UNSPECIFIED,
    ACTION_KEY_LAST,
    ACTION_KID_ARG, /* explicit or read from file during argument parsing */
    ACTION_BUILTIN_ARG
} action_key_source_t;

typedef struct action_key_keystore_spec {
    sdc_key_id_t kid;
} action_key_keystore_spec_t;

typedef struct action_key_builtin_spec {
    bool uid_set;
    bool gid_set;
    /*bool len_set; always set when selecting builtin key */
    uid_t uid;
    gid_t gid;
    sdc_key_len_t len;

    const char *permissions;

    bool secret_modifier;
    const char *modifier_filename;
} action_key_builtin_spec_t;

typedef struct action_key_spec {
    action_key_source_t source;
    action_key_keystore_spec_t keystore;
    action_key_builtin_spec_t builtin;
} action_key_spec_t;

typedef enum action_install_key_type {
    ACTION_INSTALL_KEY_UNKNOWN,
    ACTION_INSTALL_RANDOM_KEY,
    ACTION_INSTALL_PLAIN_KEY,
    ACTION_INSTALL_WRAPPED_KEY
} action_install_key_type_t;

typedef enum action_iv_mode {
    ACTION_IV_DEFAULT,
    ACTION_IV_NOIV,
    ACTION_IV_FILE,
    ACTION_IV_AUTO_GEN_NO_FILE,
    ACTION_IV_AUTO_GEN_WRITE_FILE,
} action_iv_mode_t;

typedef struct action_iv_spec {
    action_iv_mode_t iv_mode;
    size_t iv_len;
    bool iv_len_specified;
    const char *iv_filename;
} action_iv_spec_t;

typedef struct action_dgst_tag_spec {
    const char *dgst_tag_filename;
    size_t dgst_tag_len;
    bool dgst_tag_len_specified;
} action_dgst_tag_spec_t;

typedef struct action_mechanism_spec {
    char *mechanism;
    uint64_t mechanism_opt;
    bool mechanism_opt_set;
} action_mechanism_spec_t;

typedef struct action_huge_data_opt_t {
    bool enable;
    bool chunk_length_set;
    size_t chunk_length;
} action_huge_data_opt_t;

typedef struct action_option_install {
    sdc_key_storage_options_t stor_opt;
    action_install_key_type_t type;
    sdc_key_fmt_t fmt;
    sdc_key_len_t len;
    sdc_keysecret_enc_t enc;
    gid_t gid;
    char* kid_file;
    char* permissions;
} action_option_install_t;

typedef struct action_option_rng {
    bool gen;
    size_t gen_len;
} action_option_rng_t;

typedef struct action_option_arch_info {
    bool dump_arch_key_lengths;
    bool dump_wrap_alg_blk;
    bool dump_encrypt_alg_blk;
    bool dump_signverify_alg_hash;
    bool dump_digest_hash;
    bool dump_error_list;
    bool dump_error_code;
    sdc_error_t error_code;
} action_option_arch_info_t;

typedef union action_options {
    action_option_install_t install_options;
    action_option_rng_t rng_options;
    action_option_arch_info_t arch_info_options;
} action_options_t;

typedef struct action {
    const char *name;
    action_operations_t operation;
    action_key_spec_t key_spec;
    action_iv_spec_t iv_spec;
    action_dgst_tag_spec_t dgst_tag_spec;
    action_mechanism_spec_t mech_spec;
    action_huge_data_opt_t huge_data_opts;
    action_options_t options;
    const char *input;
    const char *output;

    struct action *next;
} action_t;


#endif
